#!/bin/bash

###################
# This script is used to fix an issue with pacemaker configuration in v 4.2.1 after upgrade from an earlier version (4.0.0 and below).
# The script, which is run once, must run while Pacemaker and Postgres are running.
# The server module that runs the script does not need to have the primary role with respect to the database and does not need to be the domain controller.
# The script is idempotent in that it will not make any unnecessary changes to the S Series node.
###################

versions=""
needToCheckPacemaker=1
versionToUpdate="4.2.1.5"
global_rc=0
logFile="/var/rhino/logs/pcsUpdate.log"

logAndEcho(){
    local msg="$1"
    echo "$msg"
    echo "$msg" >> "$logFile"
}

log(){
    local msg="$1"
    echo "$msg" >> "$logFile"
}

thisVersionNeedsAnUpdate()
{
   local haveVersionLowerThan421=1
   local haveVersion421=1
   local highestVersion="0.0.0.0"
   
   for s in $(psql rhinodb -U rhinodba -t -c "select title from upgrade_history" | sed -e "s/.*\([0-9]\.[0-9]\.[0-9]\.[0-9]\)/\1/" | grep -v "^$") ; do
       if [[ $haveVersion421 == 1 ]] && [[ "${s}" == "${versionToUpdate}" ]] ; then
           haveVersion421=0
       fi
       if [[ "${s}" < "${versionToUpdate}" ]] ; then
           haveVersionLowerThan421=0
       fi
       versions="${versions} ${s} "
       if [[ "${s}" > "$highestVersion" ]] ; then
           highestVersion="${s}"
       fi
       
   done
   if [[ "${highestVersion}" > "${versionToUpdate}" ]] ; then
       # "The currently installed software version is later than version ${versionToUpdate}, so no update is required."
       return 5
   elif [[ $haveVersionLowerThan421 == 0 ]] && [[ $haveVersion421 == 0 ]] ; then
       # "The S Series node was updated to software version ${versionToUpdate} from an earlier version."
       return 0
   elif [[ $haveVersionLowerThan421 == 0 ]] && [[ $haveVersion421 != 0 ]] ; then
       # "The currently installed software version is earlier than version ${versionToUpdate}, so no update is required."
       return 2   
   elif [[ $haveVersionLowerThan421 != 0 ]] && [[ $haveVersion421 == 0 ]] ; then
       # "Software version ${versionToUpdate} was directly installed on the S Series node, so no update is required."
       return 3   
   elif [[ $haveVersionLowerThan421 != 0 ]] && [[ $haveVersion421 != 0 ]] ; then
       return 4   
   fi
   
   return 1
}

pcsNeedsUpdate()
{
   local param=$1
   if [[ "a${param}" == "a" ]] ; then
      param="ms-Master"
   fi
   cibadmin --query | grep -q ${param}
   return $?
}

updatePacemaker() 
{
    local rc=0
    log "Updating pacemaker config"  
    
    if ! pcs constraint colocation remove DBMaster postgres-ms ; then
        rc=$(( $rc + 1 ))
        log "Failed constraint colocation remove DBMaster postgres-ms"
        logAndEcho "An error occurred. Contact customer support."
    fi
    if ! pcs constraint colocation remove DBSlave postgres-ms ; then
        rc=$(( $rc + 1 ))
        log "Failed constraint colocation remove DBSlave postgres-ms"
        logAndEcho "An error occurred. Contact customer support."
    fi

    if pcsNeedsUpdate postgres-monitor-5s ; then
        if ! pcs resource op delete postgres-monitor-5s ; then
            log "Failed op delete postgres-monitor-5s"
        fi
    elif ! pcs resource op delete postgres-monitor-interval-5s ; then
        log "Failed op delete postgres-monitor-interval-5s"
    fi

    if ! pcs constraint colocation add DBMaster with Promoted postgres-ms INFINITY id=colocation-DBMaster-postgres-ms-Promoted ; then
        rc=$(( $rc + 1 ))
        log "Failed constraint colocation add DBMaster"
        logAndEcho "An error occurred. Contact customer support."
    fi
    if ! pcs constraint colocation add DBSlave with Promoted postgres-ms "-10000" id=colocation-DBSlave-postgres-ms-Promoted ; then
        rc=$(( $rc + 1 ))
        log "Failed constraint colocation add DBSlave"
        logAndEcho "An error occurred. Contact customer support."
    fi
    
    if ! pcs resource op add postgres monitor interval=5s timeout=300s on-fail=restart role=Promoted ; then
        log "Failed op add postgres monitor interval=5s"
    fi
    return $rc
}

thisVersionNeedsAnUpdate
global_rc=$?

log "Found versions ${versions}"
if [[ $global_rc == 0 ]] ; then
    logAndEcho "The S Series node was updated to software version ${versionToUpdate} from an earlier version."
    logAndEcho "Checking whether the update has already been made."
    needToCheckPacemaker=0
elif [[ $global_rc == 1 ]] ; then 
    log "Could not query versions."
    logAndEcho "An error occurred. Contact customer support."
    exit 255
elif [[ $global_rc == 2 ]] ; then
    logAndEcho "The currently installed software version is earlier than version ${versionToUpdate}, so no update is required."
    exit 2
elif [[ $global_rc == 3 ]] ; then
    logAndEcho "Software version ${versionToUpdate} was directly installed on the S Series node, so no update is required."
    exit 3
elif [[ $global_rc == 4 ]] ; then
    log "No valid versions found."
    logAndEcho "An error occurred. Contact customer support."
    exit 254
elif [[ $global_rc == 5 ]] ; then
    log "The currently installed software version is later than version ${versionToUpdate}, so no update is required."
    exit 5
else 
    log "Could not find valid version state."
    logAndEcho "An error occurred. Contact customer support."
    exit 253
fi
       
if pcsNeedsUpdate ; then
    log "ID *ms-Master was found in cibadmin --query. Update is required."
    echo "Update is required."
else
    log "ID *ms-Master was not found in cibadmin --query. No update is required."
    echo "No update is required."
    exit 0
fi

updatePacemaker
global_rc=$?
if [[ "${global_rc}" != "0" ]] ; then
    log "${global_rc} updates had errors. Contact technical support."  
    echo "An error occurred. Contact customer support."
fi

if pcsNeedsUpdate ; then
    log "ID *ms-Master was found in cibadmin --query. Update is required."
    echo "Update is required."
else
    log "ID *ms-Master was not found in cibadmin --query after update."
    if [[ "a$(cibadmin --query | grep -c 'ms-Promoted')" == "a2" ]] ; then
        log "Two instances of ms-Promoted were found in cibadmin --query. No update is required."
        echo "Update completed successfully!"
        exit 0
    fi
fi

exit ${global_rc}
